using System;
using System.Data;

using gov.va.med.vbecs.Common;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{

	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>10/8/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>This class handles incoming HL7 messages from CPRS to cancel an existing pending order in VBECS.</summary>

	#endregion

	/// <summary>
	/// Class for Canceling an order
	/// </summary>
	public class CprsCancelVbecsOrder
	{
		private HL7OmgMessage _message;
		private int _vbecsOrderNumber;
		private int _cprsOrderNumber;
		private string _vistaPatientId;
		private string _orderTable;
		private DataTable dtOrderTable;

		private const string LAST_UPDATE_USER_PARAMETER_VALUE = "CPRS";
		private const string CANCEL_ORDER_STATUS_CODE = "X";
		private const string PENDING_ORDER_STATUS_CODE = "P";
		private const string ORDERED_TEST = "test";
		private const string ORDERED_COMPONENT = "comp";

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5783"> 
		///		<ExpectedInput>Valid HL7OmgMessage cancel ordered component message.</ExpectedInput>
		///		<ExpectedOutput>Non-null instance of CprsCancelVbecsOrder.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5784"> 
		///		<ExpectedInput>Null HL7OmgMessage input parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to create an instance of CprsCancelVbecsOrder and load the HL7 message containing information related to the cancel order request.
		/// </summary>
		/// <param name="message"><see cref="HL7OmgMessage"/> containing messages segments used to gather data related to the VBECS order to be canceled.</param>
		public CprsCancelVbecsOrder( HL7OmgMessage message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );

			this._message = message;
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5718"> 
		///		<ExpectedInput>Valid instance of CprsCancelVbecsOrder type with component cancel request message.</ExpectedInput>
		///		<ExpectedOutput>Non-Null HL7OrgMessage with Acknowledgement code of 'AA', Order Control code of 'CR',
		///		and OrderedComponent.OrderStatus is set to 'X'.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5719"> 
		///		<ExpectedInput>Invalid order control code of 'NW'.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5785"> 
		///		<ExpectedInput>Null VbecsOrderNumber in HL7OmgMessage input parameter.</ExpectedInput>
		///		<ExpectedOutput>Application reject HL7 acknowledgement message</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5828"> 
		///		<ExpectedInput>Null CprsOrderNumber in HL7OmgMessage input parameter.</ExpectedInput>
		///		<ExpectedOutput>Application reject HL7 acknowledgement message</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5829"> 
		///		<ExpectedInput>Valid instance of CprsCancelVbecsOrder type with diagnostic test cancel request message.</ExpectedInput>
		///		<ExpectedOutput>Non-Null HL7OrgMessage with Acknowledgement code of 'AA', Order Control code of 'CR',
		///		and OrderedTest.OrderStatus is set to 'X'.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method is responsible for validating the data in the HL7 messages and cancelling the order in VBECS.
		/// CR 2961
		/// </summary>
		/// <returns><see cref="HL7OrgMessage"/> representing the response to the cancel order request.</returns>
		public HL7OrgMessage ProcessCancelOrder()
		{
			if( this.Message == null )
				throw( new ArgumentNullException( "_message" ) );

			if( _message.OrderControlCode != "CA" )
			{
				throw( new HL7Exception( "Order Control Code must be 'CA' to call CprsCancelVbecsOrder.ProcessNewOrder" ) );
			}
			if( HL7Utility.ConvertString( this.Message.ORC[3]) != null )
			{
				this._vbecsOrderNumber = Convert.ToInt32( HL7Utility.ParseGetSegmentData( this.Message.ORC[3], this.Message.Delimiters[1].ToString(), 0 ) );
				if( this._vbecsOrderNumber < 1 )
					return CprsOrgMessage.AckError( this.Message, AckCodes.AR, "VBECS order number is required to cancel a VBECS order." );

			}

			if( HL7Utility.ConvertString( this.Message.OBR[2]) != null )
			{
				this._cprsOrderNumber = Convert.ToInt32(HL7Utility.ParseGetSegmentData( this.Message.OBR[2], this.Message.Delimiters[1].ToString(), 0 ) );
				if( this._cprsOrderNumber < 1 )
					return CprsOrgMessage.AckError( this.Message, AckCodes.AR, "CPRS order number is required to cancel a VBECS order." );

			}

			this._orderTable = HL7AL.CprsOmgMessage.GetOrderTableTypeID( HL7Utility.ConvertString(this.Message.UniversalServiceID[0]) );

			GetVistaPatientIdFromMessage();
			if( VistaPatientId == null )
			{
				return CprsOrgMessage.AckError( this.Message, AckCodes.AR, "VistA Patient ID (DFN) is required to cancel a VBECS order." );
			}

			try
			{
				if( (OrderTable == ORDERED_TEST && CanCancelOrderedTest()) || (OrderTable == ORDERED_COMPONENT && CanCancelOrderedComponent()) )
				{
					return CprsOrgMessage.AppAcceptAck( _message, OrderControlCodes.CR, this.VbecsOrderNumber );
				}

				return CprsOrgMessage.AppAcceptAck( _message, OrderControlCodes.UC, this.VbecsOrderNumber );
			}
            // *** Fortify Justified Code ***
            // *** We dont suppress messages and use _ackErrorCode = AckErrorCodes.SegmentSequenceError inside try/catch to maintain errored objects state. Outer functionality is designed to use that property to check if error occurred and handle it appropriately. ***
            catch (Exception exc)
            {
                // CR 2961
                HL7Interface intParms = new HL7Interface(InterfaceName.CPRS.ToString());

                HL7MailMessage sendMail = new HL7MailMessage();
                // CR 1928 - BNT 6/2/06 Removed HL7 Message text from email message.
                string body = "An Exception occured trying to process a cancel order request from CPRS.\n\nError Message:\n" + HL7Utility.GetInnerExceptionFromException(exc);
                sendMail.SendMessage(intParms.InterfaceAdministratorEmail, PII                   ",
                    "VBECS HL7 Exception", body, "smtp.DNS   ");

                return CprsOrgMessage.AppAcceptAck(_message, OrderControlCodes.CR, this.VbecsOrderNumber);
            }
            // *** Fortify Justified Code *** 

        }

		private bool CanCancelOrderedComponent()
		{
			dtOrderTable = HL7AL.CprsOmgMessage.GetOrderedComponentbyCPRSOrderNumber(CprsOrderNumber.ToString());

			if (dtOrderTable.Rows.Count > 0)
			{
				if( dtOrderTable.Rows[0].IsNull(TABLE.OrderedComponent.OrderStatusCode))
				{
					return false;
				}
				if( dtOrderTable.Rows[0][TABLE.OrderedComponent.OrderStatusCode].ToString() != PENDING_ORDER_STATUS_CODE )
				{
					return false;
				}
				if( dtOrderTable.Rows[0][TABLE.Patient.VistaPatientId].ToString() != VistaPatientId )
				{
					return false;
				}

				return CancelOrderedComponent(dtOrderTable);

			}
			else
			{
				return false;
			}
		}

		private bool CanCancelOrderedTest()
		{
			dtOrderTable = HL7AL.CprsOmgMessage.GetOrderedTestbyCPRSOrderNumber(CprsOrderNumber.ToString());

			if (dtOrderTable.Rows.Count > 0)
			{
				if( dtOrderTable.Rows[0].IsNull(TABLE.OrderedTest.OrderStatusCode))
				{
					return false;
				}
				if( dtOrderTable.Rows[0][TABLE.OrderedTest.OrderStatusCode].ToString() != PENDING_ORDER_STATUS_CODE )
				{
					return false;
				}
				if( dtOrderTable.Rows[0][TABLE.Patient.VistaPatientId].ToString() != VistaPatientId )
				{
					return false;
				}

				return CancelOrderedTest(dtOrderTable);
			}
			else
			{
				return false;
			}
		}

		private bool CancelOrderedComponent(DataTable dtOrder)
		{
			if( dtOrder.Rows.Count > 0 )
			{
				DataTable dtUpdateOrderedComponent = new DataTable();
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderedComponentGuid, typeof(Guid));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.PatientOrderGuid, typeof(Guid));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.ComponentClassId, typeof(int));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.CprsOrderNumber);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.VbecsOrderNumber);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.RequiredUnitQuantity, typeof(decimal));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.SpecimenStatusCode);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.PreOpIndicator, typeof(bool));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderStatusCode);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderUrgencyCode);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.RequiredDatetime, typeof(DateTime));
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.CancellationText);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.LabOrderNumber);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.DivisionCode);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.LastUpdateUser);
				dtUpdateOrderedComponent.Columns.Add(TABLE.OrderedComponent.LastUpdateFunctionId, typeof(int));

				DataRow drUpdateOrderedComponent = dtUpdateOrderedComponent.NewRow();
				drUpdateOrderedComponent[TABLE.OrderedComponent.OrderedComponentGuid] = (Guid)dtOrder.Rows[0][TABLE.OrderedComponent.OrderedComponentGuid];
				drUpdateOrderedComponent[TABLE.OrderedComponent.PatientOrderGuid] = (Guid)dtOrder.Rows[0][TABLE.OrderedComponent.PatientOrderGuid];
				drUpdateOrderedComponent[TABLE.OrderedComponent.ComponentClassId] = Convert.ToInt32(dtOrder.Rows[0][TABLE.OrderedComponent.ComponentClassId].ToString().Trim());
				drUpdateOrderedComponent[TABLE.OrderedComponent.CprsOrderNumber] = this.CprsOrderNumber.ToString();
				drUpdateOrderedComponent[TABLE.OrderedComponent.VbecsOrderNumber] = this.VbecsOrderNumber.ToString();
				drUpdateOrderedComponent[TABLE.OrderedComponent.RequiredUnitQuantity] = Convert.ToDecimal(dtOrder.Rows[0][TABLE.OrderedComponent.RequiredUnitQuantity].ToString().Trim());
				drUpdateOrderedComponent[TABLE.OrderedComponent.SpecimenStatusCode] = dtOrder.Rows[0][TABLE.OrderedComponent.SpecimenStatusCode].ToString().Trim();
				drUpdateOrderedComponent[TABLE.OrderedComponent.PreOpIndicator] = Convert.ToBoolean(dtOrder.Rows[0][TABLE.OrderedComponent.PreOpIndicator]);
				drUpdateOrderedComponent[TABLE.OrderedComponent.OrderStatusCode] = CANCEL_ORDER_STATUS_CODE;
				drUpdateOrderedComponent[TABLE.OrderedComponent.OrderUrgencyCode] = dtOrder.Rows[0][TABLE.OrderedComponent.OrderUrgencyCode].ToString().Trim();
				drUpdateOrderedComponent[TABLE.OrderedComponent.RequiredDatetime] = HL7DateFormat.ConvertHL7DateTime(HL7Utility.ConvertString( this.Message.TimingQuantity[4] ));
				drUpdateOrderedComponent[TABLE.OrderedComponent.CancellationText] = HL7Utility.ConvertString( HL7Utility.ParseGetSegmentData( this.Message.ORC[16], this.Message.Delimiters[1].ToString(), 1 ) );
				drUpdateOrderedComponent[TABLE.OrderedComponent.LabOrderNumber] = dtOrder.Rows[0][TABLE.OrderedComponent.LabOrderNumber].ToString().Trim();
				drUpdateOrderedComponent[TABLE.OrderedComponent.DivisionCode] = dtOrder.Rows[0][TABLE.OrderedComponent.DivisionCode].ToString().Trim();
				drUpdateOrderedComponent[TABLE.OrderedComponent.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
				drUpdateOrderedComponent[TABLE.OrderedComponent.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

				dtUpdateOrderedComponent.Rows.Add(drUpdateOrderedComponent);

				if(  CprsOmgMessage.UpdateVistaPatientOrder(new DataTable(), new DataTable(), new DataTable(), dtUpdateOrderedComponent) < 0 )
				{
					return false;
				}
                UpdateMessageLog(MessageStatus.SuccessfullyCompleted, Message, new HL7Interface(InterfaceName.CPRS.ToString()));
				return true;
			}
			return false;
		}

		private bool CancelOrderedTest(DataTable dtOrder)
		{
			if( dtOrder.Rows.Count > 0 )
			{
				DataTable dtUpdateOrderedTest = new DataTable();
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.PatientOrderGuid, typeof(Guid));
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.CprsOrderNumber);
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.VbecsOrderNumber);
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.OrderStatusCode);
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.CancellationText);
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser);
				dtUpdateOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));

				DataRow drUpdateOrderedTest = dtUpdateOrderedTest.NewRow();
				drUpdateOrderedTest[TABLE.OrderedTest.OrderedTestGuid] = (Guid)dtOrder.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
				drUpdateOrderedTest[TABLE.OrderedTest.PatientOrderGuid] = (Guid)dtOrder.Rows[0][TABLE.OrderedTest.PatientOrderGuid];
				drUpdateOrderedTest[TABLE.OrderedTest.CprsOrderNumber] = this.CprsOrderNumber.ToString().Trim();
				drUpdateOrderedTest[TABLE.OrderedTest.VbecsOrderNumber] = this.VbecsOrderNumber.ToString().Trim();
				drUpdateOrderedTest[TABLE.OrderedTest.OrderStatusCode] = CANCEL_ORDER_STATUS_CODE;
				drUpdateOrderedTest[TABLE.OrderedTest.CancellationText] = HL7Utility.ConvertString( HL7Utility.ParseGetSegmentData( this.Message.ORC[16], this.Message.Delimiters[1].ToString(), 1 ) );
				drUpdateOrderedTest[TABLE.OrderedTest.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
				drUpdateOrderedTest[TABLE.OrderedTest.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

				dtUpdateOrderedTest.Rows.Add(drUpdateOrderedTest);

				if( CprsOmgMessage.UpdateVistaPatientOrder(new DataTable(), new DataTable(), new DataTable(), dtUpdateOrderedTest) < 0 )
				{
					return false;
				}
                UpdateMessageLog(MessageStatus.SuccessfullyCompleted, Message, new HL7Interface(InterfaceName.CPRS.ToString()));
				return true;

			}
			return false;
		}

        //CR 3560
        private static void UpdateMessageLog(MessageStatus status, HL7ProtocolMessage message, HL7Interface intParms)
        {
            HL7MessageLog.UpdateMessageStatus(status, message.GetMessageControlID(), null, 0, intParms.InterfaceName, UpdateFunction.HL7CprsInterface);
        }

		private void GetVistaPatientIdFromMessage()
		{
			string[] patientIdList = HL7Utility.ParseGetStringArray( this.Message.PID[3], this.Message.Delimiters[2].ToString() );
			for (int i = 0; i < patientIdList.Length ; i++ )
			{
				string[] idComponents = HL7Utility.ParseGetStringArray( patientIdList[i], this.Message.Delimiters[1].ToString() );
				if( idComponents[4] == "PI" )
				{
					_vistaPatientId = HL7Utility.ConvertString( idComponents[0] );
				}
			}
		}

		#region [Properties]

		private HL7OmgMessage Message
		{
			get
			{
				return _message;
			}
		}

		private int VbecsOrderNumber
		{
			get
			{
				return _vbecsOrderNumber;
			}
		}

		private int CprsOrderNumber
		{
			get
			{
				return _cprsOrderNumber;
			}
		}

		private string VistaPatientId
		{
			get
			{
				return _vistaPatientId;
			}
		}

		private string OrderTable
		{
			get
			{
				return this._orderTable;
			}
		}


		#endregion

	}

}
